#!/bin/sh
#
# ssh_blocker for OpenBSD and Packet Filter
# Copyright (C) 2004 Juan J. Martinez <jjm*at*usebox*dot*net>
#
# Permission is hereby granted, free of charge, to any person obtaining 
# a copy of this software and associated documentation files (the "Software"),
# to deal in the Software without restriction, including without limitation
# the rights to use, copy, modify, merge, publish, distribute, sublicense,
# and/or sell copies of the Software, and to permit persons to whom the
# Software is furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
# OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
#
# http://blackshell.usebox.net/pub/shell/
#

VERSION="Version 20041211-1"

# configuration variables (DEFAULTS)
pfctl="/sbin/pfctl"
pf_table="sshscan"
authlog_path="/var/log/authlog"
white_list=" 127.0.0.1"
verbose=""
log_path=""

function usage
{
cat << USAGE
ssh_blocker for OpenBSD and Packet Filter ($VERSION)
Copyright (C) 2004 Juan J. Martinez <jjm*at*usebox*dot*net>

usage: ssh_blocker.sh -h
       ssh_blocker.sh [-t pf_table] [-a authlog_path]
                      [-w IP_for_white_list ] [-v]
                      [-l log_path ]

USAGE
}

while getopts ":ht:a:w:vl:" switch; do
	case $switch in

		h)
			usage
			exit 0
			;;

		t)
			pf_table="$OPTARG"
			;;

		a)
			if [ ! -e "$OPTARG" ]; then
				echo "$OPTARG: no such file"
				exit 1
			fi
			authlog_path="$OPTARG"
			;;

		w)
			white_list="$white_list $OPTARG"
			;;

		v)
			verbose="YES"
			;;

		l)
			if [ ! -e "$OPTARG" ]; then
				echo "$OPTARG: no such file"
				exit 1
			fi
			log_path="$OPTARG"
			;;

		*)
			echo "unknown switch, try -h"
			exit 1
			;;
	esac
done

if [ "X$log_path" = "X" ]; then
	# get the IPs of people trying illegal users from authlog
	infected_list=$(awk '/Failed\ password\ for\ (illegal|invalid)\ user/ { print $13 }' $authlog_path)
else
	if [ "X$verbose" = "XYES" ]; then
		echo "$(date '+%b %d %T'): using ssh_blocker log ($log_path) instead authlog"
	fi
	# else get IPs from log
	infected_list=$(awk '/Infected\ host\ detected/ { print $7 }' $log_path)
fi

# we want unique IPs
infected_list=$(echo "$infected_list" | sort | uniq)

# get already blocked IPs
blocked=$($pfctl -t$pf_table -T show)

for infected in $infected_list; do

	echo $white_list | grep -v $infected > /dev/null
	if [ $? = 0 ]; then

		echo $blocked | grep $infected > /dev/null
		if [ $? = 0 ]; then
			if [ "X$verbose" = "XYES" ]; then
				echo "$(date '+%b %d %T'): Infected host detected $infected - IP Address already blocked"
			fi
		else
			echo -n "$(date '+%b %d %T'): Infected host detected $infected"
			echo -n " - Adding IP Address to the blocked table: "
			$pfctl -t$pf_table -T add $infected 2>&1
		fi
	fi
done

# EOF

