#!/bin/sh
#
# ssh_blocker_wrap for tcp_wrappers
# Copyright (C) 2005 Scott Harney <scott_harney*at*yahoo*dot*com>
#
# Permission is hereby granted, free of charge, to any person obtaining 
# a copy of this software and associated documentation files (the "Software"),
# to deal in the Software without restriction, including without limitation
# the rights to use, copy, modify, merge, publish, distribute, sublicense,
# and/or sell copies of the Software, and to permit persons to whom the
# Software is furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
# OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
#
# inspired by ssh_blocker for OpenBSD by Juan Martinez
# found at http://blackshell.usebox.net/pub/shell/
#

VERSION="Version 20050103-1"

# configuration variables (DEFAULTS)
authlog_path="/var/log/auth.log"
white_list=" 127.0.0.1"
verbose=""
log_path=""
hostsdeny="/etc/hosts.deny"

function usage
{
cat << USAGE
ssh_blocker for tcp wrappers ($VERSION)
Copyright (C) 2005 Scott Harney <scott_harney*at*yahoo*dot*com

usage: ssh_blocker_wrap.sh -h
       ssh_blocker_wrap.sh [-a authlog_path] [-f hosts_deny_path]
                      [-w IP_for_white_list ] [-v]
                      [-l log_path ]

USAGE
}

while getopts ":ht:a:f:w:vl:" switch; do
	case $switch in

		h)
			usage
			exit 0
			;;

		a)
			if [ ! -e "$OPTARG" ]; then
				echo "$OPTARG: no such file"
				exit 1
			fi
			authlog_path="$OPTARG"
			;;

		w)
			white_list="$white_list $OPTARG"
			;;

		v)
			verbose="YES"
			;;

		l)
			if [ ! -e "$OPTARG" ]; then
				echo "$OPTARG: no such file"
				exit 1
			fi
			log_path="$OPTARG"
			;;

		f)
			hostsdeny="$OPTARG"
			;;

		*)
			echo "unknown switch, try -h"
			exit 1
			;;
	esac
done

if [ "X$log_path" = "X" ]; then
	# get the IPs of people trying illegal users from authlog
	infected_list=$(awk '/Failed\ password\ for\ (illegal|invalid)\ user/ { print $13 }' $authlog_path | sed 's/::ffff://') 
else
	if [ "X$verbose" = "XYES" ]; then
		echo "$(date '+%b %d %T'): using ssh_blocker log ($log_path) instead authlog"
	fi
	# else get IPs from log
	infected_list=$(awk '/Infected\ host\ detected/ { print $7 }' $log_path)
fi

# we want unique IPs
infected_list=$(echo "$infected_list" | sort | uniq )

# get already blocked IPs
if [ -f $hostsdeny ]; then
  blocked="$(awk -F: '$blocked { print $2 }'i $hostsdeny)"
fi

for infected in $infected_list; do
	echo $white_list | grep -v $infected > /dev/null
	if [ $? = 0 ]; then

		echo $blocked | grep $infected > /dev/null
		if [ $? = 0 ]; then
			if [ "X$verbose" = "XYES" ]; then
				echo "$(date '+%b %d %T'): Infected host detected $infected - IP Address already blocked"
			fi
		else
			echo -n "$(date '+%b %d %T'): Infected host detected $infected"
			echo " - Adding IP Address to the blocked table: "
			echo "sshd:$infected" >> $hostsdeny
		fi
	else
		if [ "X$verbose" = "XYES" ]; then
			echo "$(date '+%b %d %T'): Infected host detected $infected is in whitelist"
		fi
	fi
done

# EOF

